RSpec.describe Msf::Exploit::Remote::Pkcs12::Storage do

  if ENV['REMOTE_DB']
    before {skip('Not supported for remote DB')}
  end

  include_context 'Msf::DBManager'

  let(:workspace) { FactoryBot.create(:mdm_workspace) }
  let(:origin) do
    FactoryBot.create(
      :metasploit_credential_origin_service,
      service: FactoryBot.create(
        :mdm_service,
        host: FactoryBot.create(:mdm_host, workspace: workspace)
      )
    )
  end
  let(:username) { 'n00tmeg' }
  let(:realm) { 'test_realm' }
  let!(:creds) do
    creds = 2.times.map do
      FactoryBot.create(
        :metasploit_credential_core,
        private: FactoryBot.create(:metasploit_credential_pkcs12),
        origin: origin,
        workspace: workspace
      )
    end
    # Add a core credential with specific username and realm
    creds << FactoryBot.create(
      :metasploit_credential_core,
      public: FactoryBot.create(:metasploit_credential_username, username: username),
      realm: FactoryBot.create(:metasploit_credential_realm, value: realm),
      private: FactoryBot.create(:metasploit_credential_pkcs12),
      origin: origin,
      workspace: workspace
    )
    creds
  end

  subject(:storage) { described_class.new(framework: framework) }

  before :each do
    framework.db.workspace = workspace
  end

  describe '#pkcs12' do
    it 'returns an Array of StoredPkcs12' do
      expect(storage.pkcs12).to be_a(Array)
      expect(storage.pkcs12.all? {|e| e.is_a?(Msf::Exploit::Remote::Pkcs12::StoredPkcs12)}).to be true
    end

    it 'returns all the StoredPkcs12 by default' do
      expect(storage.pkcs12.size).to eq(creds.size)
      expect(storage.pkcs12.map(&:id).sort).to eq(creds.map(&:id).sort)
    end

    it 'yields each StoredPkcs12 by default' do
      expect { |b| storage.pkcs12({}, &b) }.to yield_control.exactly(creds.size).times
      results = creds.map(&:id)
      storage.pkcs12 do |stored_pkcs12|
        results.delete(stored_pkcs12.id)
      end
      expect(results).to be_empty
    end
  end

  describe '#filter_pkcs12' do
    it 'returns an Array of Metasploit::Credential::Core' do
      expect(storage.filter_pkcs12({})).to be_a(Array)
      expect(storage.filter_pkcs12({}).all? {|e| e.is_a?(Metasploit::Credential::Core)}).to be true
    end

    it 'returns all the credentials' do
      expect(storage.filter_pkcs12({}).size).to eq(creds.size)
      expect(storage.filter_pkcs12({}).map(&:id).sort).to eq(creds.map(&:id).sort)
    end

    context 'with options to match an id' do
      it 'returns the correct StoredPkcs12' do
        cred = creds.sample
        results = storage.filter_pkcs12(id: cred.id)
        expect(results.size).to eq(1)
        expect(results.first.id).to eq(cred.id)
      end
    end

    context 'with options to match an non-existing id' do
      it 'raises an ActiveRecord::RecordNotFound exception' do
        expect { storage.filter_pkcs12({ id: (creds.map(&:id).max + 1) }) }.to raise_error(ActiveRecord::RecordNotFound)
      end
    end

    context 'with options to match a username' do
      it 'returns the correct StoredPkcs12' do
        results = storage.filter_pkcs12(username: username)
        expect(results.size).to eq(1)
        expect(results.first.public.username).to eq(username)
      end
    end

    context 'with options to match a realm' do
      it 'returns the correct StoredPkcs12' do
        results = storage.filter_pkcs12(realm: realm)
        expect(results.size).to eq(1)
        expect(results.first.realm.value).to eq(realm)
      end
    end

    context 'with options to match a workspace' do
      before :each do
        workspace = FactoryBot.create(:mdm_workspace, name: 'test_workspace')
        origin = FactoryBot.create(
          :metasploit_credential_origin_service,
          service: FactoryBot.create(
            :mdm_service,
            host: FactoryBot.create(:mdm_host, workspace: workspace)
          )
        )
        creds << FactoryBot.create(
          :metasploit_credential_core,
          private: FactoryBot.create(:metasploit_credential_pkcs12),
          origin: origin,
          workspace: workspace
        )
      end
      it 'returns the correct StoredPkcs12' do
        results = storage.filter_pkcs12(workspace: 'test_workspace')
        expect(results.size).to eq(1)
        expect(results.first.workspace.name).to eq('test_workspace')
      end
    end

    context 'with a Metasploit::Credential::Password type credential' do
      before :each do
        creds << FactoryBot.create(
          :metasploit_credential_core,
          private: FactoryBot.create(:metasploit_credential_password),
          origin: origin,
          workspace: workspace
        )
      end
      it 'only returns the Metasploit::Credential::Pkcs12 credentials' do
        expect(storage.filter_pkcs12({}).size).to eq(creds.size - 1)
        expect(storage.filter_pkcs12({}).map(&:id).sort).to eq(creds[0..-2].map(&:id).sort)
      end

      context 'and using the option to macth this credential\'s ID' do
        it 'returns an empty Array' do
          expect(storage.filter_pkcs12({id: creds.last.id})).to be_empty
        end
      end
    end

    context 'with an option to match the Pkcs12 status' do

      context 'when the status is not set on any credentials' do
        it 'returns all the credentials with an option to match active status' do
          expect(storage.filter_pkcs12({status: 'active'}).size).to eq(creds.size)
          expect(storage.filter_pkcs12({status: 'active'}).map(&:id).sort).to eq(creds.map(&:id).sort)
        end

        it 'returns no credentials with an option to match inactive status' do
          expect(storage.filter_pkcs12({status: 'inactive'})).to be_empty
        end
      end

      context 'when the status is set on some credentials' do
        before :each do
          creds << FactoryBot.create(
            :metasploit_credential_core,
            private: FactoryBot.create(:metasploit_credential_pkcs12_with_status, status: 'active'),
            origin: origin,
            workspace: workspace
          )
          creds << FactoryBot.create(
            :metasploit_credential_core,
            private: FactoryBot.create(:metasploit_credential_pkcs12_with_status, status: 'inactive'),
            origin: origin,
            workspace: workspace
          )
        end

        it 'returns all the credentials with no status set and the active credentials with an option to match active status' do
          results = storage.filter_pkcs12({status: 'active'})
          expect(results.size).to eq(creds.size - 1)
          expect(results.all? {|cred| cred.private.status.nil? || cred.private.status == 'active'}).to be true
        end

        it 'returns the inactive credentials with an option to match inactive status' do
          results = storage.filter_pkcs12({status: 'inactive'})
          expect(results.size).to eq(1)
          expect(results.first.private.status).to eq('inactive')
        end
      end

    end

    context 'when a certificate `not_after` date is in the past (expired)' do
      before :each do
        priv = FactoryBot.create(
          :metasploit_credential_pkcs12,
          not_before: Time.now - 2.hours,
          not_after: Time.now - 1.hour
        )
        creds << FactoryBot.create(
          :metasploit_credential_core,
          private: priv,
          origin: origin,
          workspace: workspace
        )
      end

      it "returns all the credentials but this one" do
        expect(storage.filter_pkcs12({}).size).to eq(creds.size - 1)
        expect(storage.filter_pkcs12({}).map(&:id).sort).to eq(creds[0..-2].map(&:id).sort)
      end
    end

    context 'when the certificate not_before date is in the future' do
      before :each do
        priv = FactoryBot.create(
          :metasploit_credential_pkcs12,
          not_before: Time.now + 1.hour,
          not_after: Time.now + 2.hours
        )
        creds << FactoryBot.create(
          :metasploit_credential_core,
          private: priv,
          origin: origin,
          workspace: workspace
        )
      end

      it "returns all the credentials but this one" do
        expect(storage.filter_pkcs12({}).size).to eq(creds.size - 1)
        expect(storage.filter_pkcs12({}).map(&:id).sort).to eq(creds[0..-2].map(&:id).sort)
      end
    end
  end

  describe '#delete' do
    it 'returns an Array of StoredPkcs12' do
      results = storage.delete({})
      expect(results).to be_a(Array)
      expect(results.all? {|e| e.is_a?(Msf::Exploit::Remote::Pkcs12::StoredPkcs12)}).to be true
    end

    it 'deletes all the credentials and returns the deleted StoredPkcs12 objects' do
      results = storage.delete({})
      expect(Msf::Exploit::Remote::Pkcs12::Storage.new(framework: framework).pkcs12).to be_empty
      expect(results.size).to eq(creds.size)
      expect(results.map(&:id).sort).to eq(creds.map(&:id).sort)
    end

    context 'with options to match an id' do
      it 'deletes the matching credential and return the deleted StoredPkcs12 objects' do
        cred = creds.sample
        results = storage.delete(id: cred.id)
        remaining_pkcs12 = Msf::Exploit::Remote::Pkcs12::Storage.new(framework: framework).pkcs12
        expect(remaining_pkcs12.map(&:id).sort).to eq((creds.map(&:id) - [cred.id]).sort)
        expect(results.size).to eq(1)
        expect(results.first.id).to eq(cred.id)
      end
    end
  end

  describe '#deactivate' do
    it 'deactivates the Pkcs12 that matches the provided ID and returns the updated StoredPkcs12' do
      cred = creds.sample
      results = storage.deactivate(ids: [cred.id])
      expect(results.first.id).to eq(cred.id)
      expect(results.first.status).to eq('inactive')
      expect(framework.db.creds(id: cred.id).first.private.status).to eq('inactive')
    end
  end

  describe '#activate' do
    before :each do
      creds << FactoryBot.create(
        :metasploit_credential_core,
        private: FactoryBot.create(:metasploit_credential_pkcs12_with_status, status: 'inactive'),
        origin: origin,
        workspace: workspace
      )
    end

    it 'activates the Pkcs12 that matches the provided ID and returns the updated StoredPkcs12' do
      results = storage.activate(ids: [creds.last.id])
      expect(results.first.id).to eq(creds.last.id)
      expect(results.first.status).to eq('active')
      expect(framework.db.creds(id: creds.last.id).first.private.status).to eq('active')
    end
  end
end
